#include-once
#include <OutlookEX_Base.au3>
#include <StringConstants.au3>
; #INDEX# =======================================================================================================================
; Title .........: Microsoft Outlook Function Library (Client related)
; AutoIt Version : 3.3.10.2
; UDF Version ...: See variable $__g_sVersionOutlookEX_GUI
; Language ......: English
; Description ...: A collection of functions for accessing and manipulating Microsoft Outlook (Client related functions)
; Author(s) .....: water
; Modified.......: See variable $__g_sVersionOutlookEX_GUI
; Contributors ..:
; Resources .....: Outlook Developer Reference: https://docs.microsoft.com/en-us/office/vba/api/overview/outlook
; ===============================================================================================================================
Global $__g_sVersionOutlookEX_GUI = "OutlookEX_GUI: 1.4.0.0 2021-04-13"

#Region #VARIABLES#
; #VARIABLES# ===================================================================================================================
; See OutlookEX_Base.au3
; ===============================================================================================================================
#EndRegion #VARIABLES#

#Region #CONSTANTS#
; #CONSTANTS# ===================================================================================================================
; See OutlookEX_Base.au3
; ===============================================================================================================================
#EndRegion #CONSTANTS#

; #CURRENT# =====================================================================================================================
;_OL_BarGroupAdd
;_OL_BarGroupDelete
;_OL_BarGroupGet
;_OL_BarShortcutAdd
;_OL_BarShortcutDelete
;_OL_BarShortcutGet
;_OL_NavigationFolderAdd
;_OL_NavigationFolderGet
;_OL_NavigationFolderRemove
;_OL_NavigationGroupAdd
;_OL_NavigationGroupGet
;_OL_NavigationGroupRemove
;_OL_SelectionSet
;_OL_ViewApply
;_OL_ViewCreate
;_OL_ViewDelete
;_OL_ViewExists
;_OL_ViewFieldAdd
;_OL_ViewFontSet
;_OL_ViewFormatRuleAdd
;_OL_ViewGet
;_OL_ViewModify
;_OL_ViewPropertyGet
;_OL_ViewPropertySet
;_OL_ViewSortFieldAdd
; ===============================================================================================================================

; #INTERNAL_USE_ONLY# ===========================================================================================================
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_BarGroupAdd
; Description ...: Adds a group to the OutlookBar.
; Syntax.........: _OL_BarGroupAdd($oOL, $sGroupname[, $iPosition = 1])
; Parameters ....: $oOL        - Outlook object returned by a preceding call to _OL_Open()
;                  $oGroupname - Name of the group to be created
;                  $iPosition  - [optional] Position at which the new group will be inserted in the Shortcuts pane (default = 1 = at the top of the bar)
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - Error accessing the Outlookbar pane. For details please see @extended
;                  |2 - Error creating the group. For details please see @extended
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_BarGroupAdd($oOL, $sGroupname, $iPosition = 1)
	If $iPosition = Default Then $iPosition = 1
	Local $oPane = $oOL.ActiveExplorer.Panes("OutlookBar")
	If @error Then Return SetError(1, @error, 0)
	$oPane.Contents.Groups.Add($sGroupname, $iPosition)
	If @error Then Return SetError(2, @error, 0)
	Return 1
EndFunc   ;==>_OL_BarGroupAdd

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_BarGroupDelete
; Description ...: Deletes a group from the OutlookBar.
; Syntax.........: _OL_BarGroupDelete($oOL, $vGroupname)
; Parameters ....: $oOL        - Outlook object returned by a preceding call to _OL_Open()
;                  $vGroupname - Name or 1-based index value of the group to be deleted
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vGroupname is empty
;                  |2 - Error accessing the specified group. For details please see @extended
;                  |3 - Error removing the specified group. For details please see @extended
; Author ........: water
; Modified ......:
; Remarks .......: To delete a group by name isn't possible in Outlook 2002
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_BarGroupDelete($oOL, $vGroupname)
	If StringStripWS($vGroupname, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
	Local $oGroups = $oOL.ActiveExplorer.Panes.Item("OutlookBar").Contents.Groups
	If @error Then Return SetError(2, @error, 0)
	$oGroups.Remove($vGroupname)
	If @error Then Return SetError(3, @error, 0)
	Return 1
EndFunc   ;==>_OL_BarGroupDelete

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_BarGroupGet
; Description ...: Returns all groups in the OutlookBar.
; Syntax.........: _OL_BarGroupGet($oOL)
; Parameters ....: $oOL    - Outlook object returned by a preceding call to _OL_Open()
; Return values .: Success - two-dimensional one based array with the following information:
;                  |0 - Display name of the group
;                  |1 - $OlOutlookBarViewType constant representing the view type of the group
;                  Failure - Returns "" and sets @error:
;                  |1 - Error accessing the Outlookbar pane. For details please see @extended
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_BarGroupGet($oOL)
	Local $iIndex = 1
	Local $oPane = $oOL.ActiveExplorer.Panes("OutlookBar")
	If @error Then Return SetError(1, @error, "")
	Local $aGroups[$oPane.Contents.Groups.Count + 1][2]
	For $oGroup In $oPane.Contents.Groups
		$aGroups[$iIndex][0] = $oGroup.Name
		$aGroups[$iIndex][1] = $oGroup.ViewType
		$iIndex += 1
	Next
	$aGroups[0][0] = $iIndex - 1
	$aGroups[0][1] = UBound($aGroups, 2)
	Return $aGroups
EndFunc   ;==>_OL_BarGroupGet

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_BarShortcutAdd
; Description ...: Adds a shortcut to a group in the OutlookBar.
; Syntax.........: _OL_BarShortcutAdd($oOL, $vGroupname, $sShortcutname, $sTarget[, $iPosition = 1[, $sIcon = ""]])
; Parameters ....: $oOL           - Outlook object returned by a preceding call to _OL_Open()
;                  $vGroupname    - Name or 1-based index value of the group where the shortcut will be created
;                  $oShortcutname - Name of the shortcut to be created
;                  $sTarget       - Target of the shortcut being created. Can be an Outlook folder, filesystem folder, filesystem path or URL
;                  $iPosition     - [optional] Position at which the new shortcut will be inserted in the group (default = 0 = first shortcut in an empty group)
;                  $sIcon         - [optional] The path of the icon file e.g. C:\temp\sample.ico (default = no icon)
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - Error accessing the group specified by $vGroupname. For details please see @extended
;                  |2 - Error creating the shortcut. For details please see @extended
;                  |3 - Specified icon file could not be found
;                  |4 - Error setting the icon for the created shortcut. For details please see @extended
; Author ........: water
; Modified ......:
; Remarks .......: Specify $iPosition = 1 to position a shortcut at the top of a non empty group
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_BarShortcutAdd($oOL, $vGroupname, $sShortcutname, $sTarget, $iPosition = 0, $sIcon = "")
	If $iPosition = Default Then $iPosition = 0
	If $sIcon = Default Then $sIcon = ""
	Local $oGroup = $oOL.ActiveExplorer.Panes.Item("OutlookBar").Contents.Groups.Item($vGroupname)
	If @error Or Not IsObj($oGroup) Then Return SetError(1, @error, 0)
	Local $oShortCut = $oGroup.Shortcuts.Add($sTarget, $sShortcutname, $iPosition)
	If @error Then Return SetError(2, @error, 0)
	If $sIcon <> "" Then
		If Not FileExists($sIcon) Then Return SetError(3, 0, 0)
		$oShortCut.SetIcon($sIcon)
		If @error Then Return SetError(4, @error, 0)
	EndIf
	Return 1
EndFunc   ;==>_OL_BarShortcutAdd

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_BarShortcutDelete
; Description ...: Deletes a Shortcut from the OutlookBar.
; Syntax.........: _OL_BarShortcutDelete($oOL, $vGroupname, $vShortcutname)
; Parameters ....: $oOL           - Outlook object returned by a preceding call to _OL_Open()
;                  $vGroupname    - Name or 1-based index value of the group from where the shortcut will be deleted
;                  $vShortcutname - Name or 1-based index value of the shortcut to be deleted
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vShortcutname is empty
;                  |2 - $vGroupname is empty
;                  |3 - Error accessing the specified group. For details please see @extended
;                  |4 - Error removing the specified Shortcut. For details please see @extended
; Author ........: water
; Modified ......:
; Remarks .......: Deleting a shortcut by name isn't possible in Outlook 2002
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_BarShortcutDelete($oOL, $vGroupname, $vShortcutname)
	If StringStripWS($vShortcutname, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
	If StringStripWS($vGroupname, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(2, 0, 0)
	Local $oGroup = $oOL.ActiveExplorer.Panes.Item("OutlookBar").Contents.Groups.Item($vGroupname)
	If @error Then Return SetError(3, @error, 0)
	$oGroup.Shortcuts.Remove($vShortcutname)
	If @error Then Return SetError(4, @error, 0)
	Return 1
EndFunc   ;==>_OL_BarShortcutDelete

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_BarShortcutGet
; Description ...: Returns all shortcuts of a group in the OutlookBar.
; Syntax.........: _OL_BarShortcutGet($oOL, $vGroup)
; Parameters ....: $oOL    - Outlook object returned by a preceding call to _OL_Open()
;                  $vGroup - Name or 1-based index value of the group
; Return values .: Success - two-dimensional one based array with the following information:
;                  |0 - Display name of the shortcut
;                  |1 - Variant indicating the target of the specified shortcut in a Shortcuts pane group
;                  Failure - Returns "" and sets @error:
;                  |1 - $vGroup is empty
;                  |2 - Error accessing the specified group. For details please see @extended
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........: http://msdn.microsoft.com/en-us/library/bb176723(v=office.12).aspx
; Example .......: Yes
; ===============================================================================================================================
Func _OL_BarShortcutGet($oOL, $vGroup)
	Local $iIndex = 1
	If StringStripWS($vGroup, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, "")
	Local $oGroup = $oOL.ActiveExplorer.Panes.Item("OutlookBar").Contents.Groups.Item($vGroup)
	If @error Then Return SetError(2, @error, "")
	Local $aShortcuts[$oGroup.Shortcuts.Count + 1][2]
	For $oShortCut In $oGroup.Shortcuts
		$aShortcuts[$iIndex][0] = $oShortCut.Name
		$aShortcuts[$iIndex][1] = $oShortCut.Target
		$iIndex += 1
	Next
	$aShortcuts[0][0] = $iIndex - 1
	$aShortcuts[0][1] = UBound($aShortcuts, 2)
	Return $aShortcuts
EndFunc   ;==>_OL_BarShortcutGet

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_NavigationFolderAdd
; Description ...: Adds navigation folders to a navigation group
; Syntax.........: _OL_NavigationFolderAdd($oOL, $aNavigationFolders[, $iStartIndex = 0[, $iModuleType = Default]])
; Parameters ....: $oOL                - Outlook object returned by a preceding call to _OL_Open()
;                  $aNavigationFolders - Array holding the navigation folders to be added to the navigation group. Please see Remarks for further information
;                  $iStartIndex        - [optional] Index of the first item of $aNavigationFolders to be processed. Should be 0 or 1 (default = 0)
;                  $iModuleType        - [optional] Type of the navigation module. Can be any of the OlNavigationModuleType enumeration (default = $olModuleCalendar)
; Return values .: Success - Number of processed navigation folders
;                  Failure - Returns 0 and sets @error:
;                  |1 - $oOL is not an object
;                  |2 - Error accessing the navigation pane. @extended is set to the COM error code
;                  |3 - Error accessing the navigation module. @extended is set to the COM error code
;                  |4nnn - Error accessing the navigation groups. @extended is set to the COM error code. nnn is the index of the row in error
;                  |5nnn - The specified navigation group could not be found. @extended is set to the COM error code. nnn is the index of the row in error
;                  |6nnn - Error accessing the specified folder. @extended is set to @error as returned by _OL_FolderAccess. nnn is the index of the row in error
;                  |7nnn - Error adding the nagivation folder. @extended is set to the COM error code. nnn is the index of the row in error
;                  |8nnn - Error setting the position of the navigation folder. Must be a number and should be <= actual number of navigation folders + 1. nnn is the index of the row in error
; Author ........: water
; Modified ......:
; Remarks .......: This function only works for Outlook 2007 and later.
;                  The array has to have the same format as the result returned by _OL_NavigationFolderGet
;                  (one or zero based two-dimensional array holding Navigation group, Folder name, Folder path, IsSelected?, IsRemovable?, IsSideBySide?, Position).
;                  Folder Path needs to be in the format as expected by _OL_FolderAccess.
; Related .......: _OL_NavigationFolderRemove, _OL_NavigationFolderGet
; Link ..........: https://docs.microsoft.com/en-us/office/vba/outlook/how-to/navigation/add-a-folder-to-the-favorite-folders-group; https://www.autoitscript.com/forum/topic/158418-outlookex-udf-help-support-iii/?do=findComment&comment=1344333
; Example .......: Yes
; ===============================================================================================================================
Func _OL_NavigationFolderAdd($oOL, $aNavigationFolders, $iStartIndex = 0, $iModuleType = Default)
	Local $oPane, $oModule, $oGroups, $oGroup, $iSet = 0, $oNavigationFolder, $aFolder
	If Not IsObj($oOL) Then Return SetError(1, 0, 0)
	If $iModuleType = Default Then $iModuleType = $olModuleCalendar
	$oPane = $oOL.ActiveExplorer.NavigationPane
	If @error Then Return SetError(2, @error, 0)
	$oModule = $oPane.Modules.GetNavigationModule($iModuleType)
	If @error Then Return SetError(3, @error, 0)
	For $i = $iStartIndex To UBound($aNavigationFolders, 1) - 1
		If StringStripWS($aNavigationFolders[$i][2], $STR_STRIPALL) = "" Then ContinueLoop ; Ignore empty rows
		; Access the folder
		#ignorefunc _OL_FolderAccess
		$aFolder = _OL_FolderAccess($oOL, $aNavigationFolders[$i][2])
		If @error Then Return SetError(6000 + $i, @error, 0)
		$oGroups = $oModule.NavigationGroups
		If @error Then Return SetError(4000 + $i, @error, 0)
		$oGroup = $oGroups.Item($aNavigationFolders[$i][0]) ; Navigation group
		If @error Then Return SetError(5000 + $i, @error, 0)
		$oNavigationFolder = $oGroup.NavigationFolders.Add($aFolder[1])
		If @error Then Return SetError(7000 + $i, @error, 0)
		$oNavigationFolder.DisplayName = $aNavigationFolders[$i][1] ; DisplayName
		$oNavigationFolder.IsSelected = $aNavigationFolders[$i][4] ; If True select the navigation folder for display
		$oNavigationFolder.IsSideBySide = $aNavigationFolders[$i][5] ; If True the navigation folder is displayed in side-by-side else in overlay mode
		$oNavigationFolder.Position = $aNavigationFolders[$i][6] ; Ordinal position of the navigation folder in the navigation pane
		If @error Then Return SetError(8000 + $i, @error, 0)
		$iSet += 1
	Next
	Return $iSet
EndFunc   ;==>_OL_NavigationFolderAdd

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_NavigationFolderGet
; Description ...: Returns information about navigation folders in a group or of all groups of a navigation module
; Syntax.........: _OL_NavigationFolderGet($oOL, [$iModuleType = $olModuleCalendar[, $sNavigationGroup = Default]])
; Parameters ....: $oOL              - Outlook object returned by a preceding call to _OL_Open()
;                  $iModuleType      - [optional] Type of the navigation module. Can be any of the OlNavigationModuleType enumeration (default = $olModuleCalendar)
;                  $sNavigationGroup - [optional] Name of the navigation group to process (default = "" = process all groups in the navigation module)
; Return values .: Success - two-dimensional one based array with the following information:
;                  |0 - Name of the navigation group the navigation folder is a member of
;                  |1 - Displayname of the navigation folder
;                  |2 - Folderpath to the "real" folder. Is empty if you do not have the needed permission to access this information
;                  |3 - True indicates that the navigation folder is selected for display
;                  |4 - True indicates that the navigation folder can be removed from the navigation pane
;                  |5 - Indicates that the navigation folder is displayed in side-by-side (True) or overlay mode (False)
;                  |6 - Integer that represents the ordinal position of the navigation folder in the navigation pane
;                  Failure - Returns "" and sets @error:
;                  |1 - $oOL is not an object
;                  |2 - Error accessing the navigation pane. @extended is set to the COM error code
;                  |3 - Error accessing the navigation module. @extended is set to the COM error code
;                  |4 - Error accessing the navigation groups. @extended is set to the COM error code
;                  |5 - The specified navigation group could not be found
; Author ........: water
; Modified ......:
; Remarks .......: This function only works for Outlook 2007 and later.
; Related .......: _OL_NavigationFolderAdd, _OL_NavigationFolderRemove
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_NavigationFolderGet($oOL, $iModuleType = Default, $sNavigationGroup = Default)
	Local $oGroup, $oPane, $oModule, $iGroups, $aFolders[1][7], $j = 1, $bFound = False
	If Not IsObj($oOL) Then Return SetError(1, 0, "")
	If $iModuleType = Default Then $iModuleType = $olModuleCalendar
	If $sNavigationGroup = Default Then $sNavigationGroup = ""
	$oPane = $oOL.ActiveExplorer.NavigationPane
	If @error Then Return SetError(2, @error, "")
	$oModule = $oPane.Modules.GetNavigationModule($iModuleType)
	If @error Then Return SetError(3, @error, "")
	$iGroups = $oModule.NavigationGroups.Count
	For $i = 1 To $iGroups
		SetError(0)
		$oGroup = $oModule.NavigationGroups.Item($i)
		If @error Then Return SetError(4, @error, "")
		ReDim $aFolders[UBound($aFolders, 1) + $oGroup.NavigationFolders.Count][UBound($aFolders, 2)]
		If $sNavigationGroup = "" Or ($sNavigationGroup <> "" And $oGroup.Name = $sNavigationGroup) Then
			$bFound = True
			For $oFolder In $oGroup.NavigationFolders
				$aFolders[$j][0] = $oGroup.Name
				$aFolders[$j][1] = $oFolder.DisplayName
				$aFolders[$j][2] = $oFolder.Folder.Folderpath
				$aFolders[$j][3] = $oFolder.IsSelected
				$aFolders[$j][4] = $oFolder.IsRemovable
				$aFolders[$j][5] = $oFolder.IsSideBySide
				$aFolders[$j][6] = $oFolder.Position
				$j = $j + 1
			Next
		EndIf
	Next
	If $bFound = False Then Return SetError(5, 0, "")
	ReDim $aFolders[$j][UBound($aFolders, 2)]
	$aFolders[0][0] = $j - 1
	$aFolders[0][1] = UBound($aFolders, 2)
	Return $aFolders
EndFunc   ;==>_OL_NavigationFolderGet

#Region ==> Noch offen

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_NavigationFolderRemove
; Description ...: Removes a navigation folder from a navigation group
; Syntax.........: _OL_NavigationFolderRemove($oOL, $iModuleType, $vNavigationGroup, $vNavigationFolder)
; Parameters ....: $oOL               - Outlook object returned by a preceding call to _OL_Open()
;                  $iModuleType       - Type of the navigation module. Can be any of the OlNavigationModuleType enumeration exept $olModuleMail
;                  $vNavigationGroup  - Name or Index of the navigation group to be processed. Index is an integer that represents the ordinal position of the navigation group in the navigation pane
;                  $vNavigationFolder - Name or Index of the navigation folder to be removed. Index is an integer that represents the ordinal position of the navigation folder in the navigation group
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - $oOL is not an object
;                  |2 - Error accessing the navigation pane. @extended is set to the COM error code
;                  |3 - Error accessing the navigation module. @extended is set to the COM error code
;                  |4 - Error accessing the navigation groups. @extended is set to the COM error code
;                  |5 - The specified navigation group could not be found. @extended is set to the COM error code
;                  |6 - The specified navigation folder could not be found. @extended is set to the COM error code
;                  |7 - Error removing the specified navigation folder. @extended is set to the COM error code
;                  |8 - Error accessing the navigation folder. @extended is set to the COM error code
;                  |9 - Can't remove the navigation folder. Property IsRemovable is set to False.
; Author ........: water
; Modified ......:
; Remarks .......: Only removable folders, NavigationFolder objects with an IsRemovable property value set to True, can be removed
;                  from a NavigationFolders collection.
;                  This means that you can remove shared folders, public folders, and linked folders.
;                  However, you must use function _OL_FolderDelete to remove any user-created folders
; Related .......: _OL_NavigationFolderAdd, _OL_NavigationFolderGet
; Link ..........: https://docs.microsoft.com/en-us/office/vba/outlook/how-to/navigation/add-a-folder-to-the-favorite-folders-group
; Example .......: Yes
; ===============================================================================================================================
Func _OL_NavigationFolderRemove($oOL, $iModuleType, $vNavigationGroup, $vNavigationFolder)
	Local $oGroup, $oPane, $oModule, $oFolder, $iGroups, $iFolders, $bFound = False
	If Not IsObj($oOL) Then Return SetError(1, 0, 0)
	; Find specified navigation group
	$oPane = $oOL.ActiveExplorer.NavigationPane
	If @error Then Return SetError(2, @error, 0)
	$oModule = $oPane.Modules.GetNavigationModule($iModuleType)
	If @error Then Return SetError(3, @error, 0)
	If IsInt($vNavigationGroup) Then
		$oGroup = $oModule.NavigationGroups.Item($vNavigationGroup)
		If @error Then Return SetError(4, @error, 0)
	Else
		$iGroups = $oModule.NavigationGroups.Count
		For $i = 1 To $iGroups
			SetError(0)
			$oGroup = $oModule.NavigationGroups.Item($i)
			If @error Then Return SetError(4, @error, 0)
			If $oGroup.Name = $vNavigationGroup Then
				$bFound = True
				ExitLoop
			EndIf
		Next
		If $bFound = False Then Return SetError(5, 0, 0)
	EndIf
	; Find specified navigation folder
	If IsInt($vNavigationFolder) Then
		If $oGroup.NavigationFolders.Item($vNavigationFolder).IsRemovable <> True Then Return SetError(9, 0, 0)
		$oGroup.NavigationFolders.Delete($oGroup.NavigationFolders.Item($vNavigationFolder))
		If @error Then Return SetError(7, @error, 0)
	Else
		$iFolders = $oGroup.NavigationFolders.Count
		For $i = 1 To $iFolders
			SetError(0)
			$oFolder = $oGroup.NavigationFolders.Item($i)
			If @error Then Return SetError(8, @error, 0)
			If $oFolder.DisplayName = $vNavigationFolder Then
				If $oFolder.IsRemovable <> True Then Return SetError(9, 0, 0)
				$oGroup.NavigationFolders.Remove($oFolder)
				If @error Then Return SetError(7, @error, 0)
				$bFound = True
				ExitLoop
			EndIf
		Next
		If $bFound = False Then Return SetError(6, 0, 0)
	EndIf
	Return 1
EndFunc   ;==>_OL_NavigationFolderRemove

#EndRegion ==> Noch offen

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_NavigationGroupAdd
; Description ...: Adds a navigation group to a navigation module
; Syntax.........: _OL_NavigationGroupAdd($oOL, $sGroupname[, $iModuleType = Default]])
; Parameters ....: $oOL         - Outlook object returned by a preceding call to _OL_Open()
;                  $sGroupname  - Displayname of the navigation group. Please see Remarks for further information
;                  $iModuleType - [optional] Type of the navigation module. Can be any of the OlNavigationModuleType enumeration exept $olModuleMail (default = $olModuleCalendar)
; Return values .: Success - Object of the created navigation group
;                  Failure - Returns 0 and sets @error:
;                  |1 - $oOL is not an object
;                  |2 - $sGroupName is empty
;                  |3 - Error accessing the navigation pane. @extended is set to the COM error code
;                  |4 - Error accessing the navigation module. @extended is set to the COM error code
;                  |5 - Error accessing the navigation groups. @extended is set to the COM error code
;                  |6 - The specified navigation group could not be create. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......: This function only works for Outlook 2007 and later.
;                  An error occurs if you add more than 50 navigation groups to a NavigationGroup collection, or if you add a NavigationGroup object to the NavigationGroup collection of a MailModule object
; Related .......: _OL_NavigationGroupGet, _OL_NavigationGroupRemove
; Link ..........: https://docs.microsoft.com/en-us/office/vba/api/outlook.navigationgroups.create; https://www.autoitscript.com/forum/topic/158418-outlookex-udf-help-support-iii/?do=findComment&comment=1344333
; Example .......: Yes
; ===============================================================================================================================
Func _OL_NavigationGroupAdd($oOL, $sGroupname, $iModuleType = Default)
	Local $oPane, $oModule, $oGroups, $oGroup
	If Not IsObj($oOL) Then Return SetError(1, 0, 0)
	If StringStripWS($sGroupname, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(2, 0, 0)
	If $iModuleType = Default Then $iModuleType = $olModuleCalendar
	$oPane = $oOL.ActiveExplorer.NavigationPane
	If @error Then Return SetError(3, @error, 0)
	$oModule = $oPane.Modules.GetNavigationModule($iModuleType)
	If @error Then Return SetError(4, @error, 0)
	$oGroups = $oModule.NavigationGroups
	If @error Then Return SetError(5, @error, 0)
	$oGroup = $oGroups.Create($sGroupname)
	If @error Then Return SetError(6, @error, 0)
	Return $oGroup
EndFunc   ;==>_OL_NavigationGroupAdd

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_NavigationGroupGet
; Description ...: Returns information about navigation groups of a navigation module
; Syntax.........: _OL_NavigationGroupGet($oOL, [$iModuleType = $olModuleCalendar[, $sNavigationGroup = ""]])
; Parameters ....: $oOL              - Outlook object returned by a preceding call to _OL_Open()
;                  $iModuleType      - [optional] Type of the navigation module. Can be any of the OlNavigationModuleType enumeration (default = $olModuleCalendar)
;                  $sNavigationGroup - [optional] Name of the navigation group to process (default = "" = process all groups in the navigation module)
; Return values .: Success - two-dimensional one based array with the following information:
;                  |0 - Name of the navigation group
;                  |1 - Grouptype (Integer) as described in the OlGrouptype enumeration
;                  |2 - Integer that represents the ordinal position of the navigation group in the navigation pane
;                  Failure - Returns "" and sets @error:
;                  |1 - $oOL is not an object
;                  |2 - Error accessing the navigation pane. @extended is set to the COM error code
;                  |3 - Error accessing the navigation module. @extended is set to the COM error code
;                  |4 - Error accessing the navigation group. @extended is set to the COM error code
;                  |5 - The specified navigation group could not be found
; Author ........: water
; Modified ......:
; Remarks .......: This function only works for Outlook 2007 and later.
; Related .......: _OL_NavigationGroupAdd, _OL_NavigationGroupRemove
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_NavigationGroupGet($oOL, $iModuleType = Default, $sNavigationGroup = Default)
	Local $oGroup, $oPane, $oModule, $iGroups, $aNavigationGroups[1][3], $j = 1, $bFound = False
	If Not IsObj($oOL) Then Return SetError(1, 0, "")
	If $iModuleType = Default Then $iModuleType = $olModuleCalendar
	If $sNavigationGroup = Default Then $sNavigationGroup = ""
	$oPane = $oOL.ActiveExplorer.NavigationPane
	If @error Then Return SetError(2, @error, "")
	$oModule = $oPane.Modules.GetNavigationModule($iModuleType)
	If @error Then Return SetError(3, @error, "")
	$iGroups = $oModule.NavigationGroups.Count
	For $i = 1 To $iGroups
		SetError(0)
		$oGroup = $oModule.NavigationGroups.Item($i)
		If @error Then Return SetError(4, @error, "")
		ReDim $aNavigationGroups[$iGroups + 1][3]
		If $sNavigationGroup = "" Or ($sNavigationGroup <> "" And $oGroup.Name = $sNavigationGroup) Then
			$bFound = True
			$aNavigationGroups[$j][0] = $oGroup.Name
			$aNavigationGroups[$j][1] = $oGroup.GroupType
			$aNavigationGroups[$j][2] = $oGroup.Position
			$j = $j + 1
		EndIf
	Next
	If $bFound = False Then Return SetError(5, 0, "")
	ReDim $aNavigationGroups[$j][3]
	$aNavigationGroups[0][0] = $j - 1
	$aNavigationGroups[0][1] = 3
	Return $aNavigationGroups
EndFunc   ;==>_OL_NavigationGroupGet

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_NavigationGroupRemove
; Description ...: Removes a navigation group from a navigation module
; Syntax.........: _OL_NavigationGroupRemove($oOL, $iModuleType, $vNavigationGroup)
; Parameters ....: $oOL              - Outlook object returned by a preceding call to _OL_Open()
;                  $iModuleType      - Type of the navigation module you want to remove the navigation group from. Can be any of the OlNavigationModuleType enumeration except olModuleMail
;                  $vNavigationGroup - Name or index of the navigation group to remove. Index is an integer that represents the ordinal position of the navigation group in the navigation pane
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - $oOL is not an object
;                  |2 - Error accessing the navigation pane. @extended is set to the COM error code
;                  |3 - Error accessing the specified navigation module. @extended is set to the COM error code
;                  |4 - Error removing the specified navigation group. @extended is set to the COM error code
;                  |5 - The specified navigation group could not be found
;                  |6 - Error accessing a navigation group. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......: The function returns an error if:
;                  * The specified navigation group specified contains navigation folders in its NavigationFolders collection
;                  * The GroupType property of the specified navigation group is set to olMyFoldersGroup
;                  * The parent of the NavigationGroups collection is a MailModule object
; Related .......: _OL_NavigationGroupAdd, _OL_NavigationGroupGet
; Link ..........: https://docs.microsoft.com/en-us/office/vba/outlook/how-to/navigation/add-a-Group-to-the-favorite-Groups-group
; Example .......: Yes
; ===============================================================================================================================
Func _OL_NavigationGroupRemove($oOL, $iModuleType, $vNavigationGroup)
	Local $oGroup, $oPane, $oModule, $iGroups, $bFound = False
	If Not IsObj($oOL) Then Return SetError(1, 0, 0)
	$oPane = $oOL.ActiveExplorer.NavigationPane
	If @error Then Return SetError(2, @error, 0)
	$oModule = $oPane.Modules.GetNavigationModule($iModuleType)
	If @error Then Return SetError(3, @error, 0)
	If IsInt($vNavigationGroup) Then
		$oModule.NavigationGroups.Delete($oModule.NavigationGroups.Item($vNavigationGroup))
		If @error Then Return SetError(4, @error, 0)
	Else
		$iGroups = $oModule.NavigationGroups.Count
		For $i = 1 To $iGroups
			SetError(0)
			$oGroup = $oModule.NavigationGroups.Item($i)
			If @error Then Return SetError(6, @error, 0)
			If $oGroup.Name = $vNavigationGroup Then
				$oModule.NavigationGroups.Delete($oGroup)
				If @error Then Return SetError(4, @error, 0)
				$bFound = True
				ExitLoop
			EndIf
		Next
		If $bFound = False Then Return SetError(5, 0, 0)
	EndIf
	Return 1
EndFunc   ;==>_OL_NavigationGroupRemove

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_SelectionSet
; Description ...: Select or unselect a single, selected or all items of the active or a selected Explorer.
; Syntax.........: _OL_SelectionSet($oOL, $vItem[, $iAction = 1[, $oExplorer = Default]])
; Parameters ....: $oOL       - Outlook object returned by a preceding call to _OL_Open()
;                  $vItem     - Item to be processed. Can be:
;                  | object - Object of the item to be selected/unselected according to parameter $iAction
;                  | -1     - Select or unselect all items according to parameter $iAction
;                  $iAction   - [optional] Specifies how to process $vItem. 1 - Select the item(s), 2 - Unselect the item(s) (default = 1)
;                  $oExplorer - [optional] Explorer to process (default = active Explorer)
;                  $iBase     - [optional] If $vItem is a two-dimensional array this parameter specifies the index of the first row to process. Has to be 0 or 1 (default = 0)
;                  $iIDColumn - [optional] Zero based index of the column holding the EntryID or object of the item to select (default = 0)
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vItem is invalid. Has to be an object, a 2D array, 0 or 1
;                  |2 - Error selecting the item. @extended is set to the COM error code
;                  |3 - Error selecting all items of the view. @extended is set to the COM error code
;                  |4 - Error unselecting the item. @extended is set to the COM error code
;                  |5 - Error unselecting all items. @extended is set to the COM error code
;                  |6 - $iAction is invalid. Has to be 1 or 2
;                  |7 - Error getting item object from EntryID. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......: A selected item does not get unselected if you select it again.
;                  You can pass the result of _OL_ItemFind to select all found items.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_SelectionSet($oOL, $vItem, $iAction = Default, $oExplorer = Default, $iBase = Default, $iIDColumn = Default)
	Local $oItem, $iStoreID
	If $iAction = Default Then $iAction = 1
	If $oExplorer = Default Then $oExplorer = $oOL.ActiveExplorer()
	If $iBase = Default Then $iBase = 1
	If $iIDColumn = Default Then $iIDColumn = 0
	$iStoreID = $oExplorer.CurrentFolder.StoreID
	Switch $iAction
		Case 1 ; Select
			Select
				Case IsObj($vItem) ; Select single item
					$oExplorer.AddToSelection($vItem)
					If @error Then Return SetError(2, @error, 0)
				Case $vItem = -1 ; Select all items of the view
					$oExplorer.SelectAllItems()
					If @error Then Return SetError(3, @error, 0)
				Case IsArray($vItem) And UBound($vItem, 0) = 2 ; Select all items passed in a 2D array as EntryID or object
					For $i = $iBase To UBound($vItem, 1) - 1
						If IsObj($vItem[$i][$iIDColumn]) Then
							$oExplorer.AddToSelection($vItem[$i][$iIDColumn])
						Else
							$oItem = $oOL.Session.GetItemFromID($vItem[$i][$iIDColumn], $iStoreID)
							If @error Then Return SetError(7, @error, 0)
							$oExplorer.AddToSelection($oItem)
						EndIf
						If @error Then Return SetError(2, @error, 0)
					Next
				Case Else
					Return SetError(1, @error, 0)
			EndSelect
		Case 2 ; Unselect
			Select
				Case IsObj($vItem) ; Unselect a single item
					$oExplorer.RemoveFromSelection($vItem)
					If @error Then Return SetError(4, @error, 0)
				Case $vItem = -1 ; Unselect all selected items
					$oExplorer.ClearSelection()
					If @error Then Return SetError(5, @error, 0)
				Case IsArray($vItem) And UBound($vItem, 0) = 2 ; Unselect all items passed in a 2D array as EntryID or object
					For $i = $iBase To UBound($vItem, 1) - 1
						If IsObj($vItem[$i][$iIDColumn]) Then
							$oExplorer.RemoveFromSelection($vItem[$i][$iIDColumn])
						Else
							$oItem = $oOL.Session.GetItemFromID($vItem[$i][$iIDColumn], $iStoreID)
							If @error Then Return SetError(7, @error, 0)
							$oExplorer.RemoveFromSelection($oItem)
						EndIf
						If @error Then Return SetError(2, @error, 0)
					Next
				Case Else
					Return SetError(1, @error, 0)
			EndSelect
		Case Else
			Return SetError(6, @error, 0)
	EndSwitch
	Return 1
EndFunc   ;==>_OL_SelectionSet

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewApply
; Description ...: Applies a view to the Microsoft Outlook user interface of the specified folder.
; Syntax.........: _OL_ViewApply($oOL, $vView[, $vFolder = Default])
; Parameters ....: $oOL     - Outlook object returned by a preceding call to _OL_Open()
;                  $vView   - Index, name or object of the view to be applied
;                  $vFolder - [optional] Folder object as returned by _OL_FolderAccess or full name of folder. Has to be specified when $vView is not a view object
; Return values .: Success - Object of the view
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vFolder has not been specified
;                  |2 - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3 - Error accessing the folders views. @extended is set to the COM error code
;                  |4 - Error applying the view. @extended is set to the COM error code
;                  |5 - View does not exist or an error occurred. @extended is set to the COM error code (if any)
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewApply($oOL, $vView, $vFolder = Default)
	Local $oView, $oViews
	If IsObj($vView) Then
		$oView = $vView
	Else
		If Not IsObj($vFolder) Then
			If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
			Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
			If @error Then Return SetError(2, @error, 0)
			$vFolder = $aFolder[1]
		EndIf
		$oViews = $vFolder.Views
		If @error Then Return SetError(3, @error, 0)
		; Check if the view exist
		$oView = $oViews.Item($vView)
		If @error Or (Not IsObj($oView)) Then Return SetError(5, @error, 0)
	EndIf
	; Apply view
	$oView.Apply()
	If @error Then Return SetError(4, @error, 0)
	Return $oView
EndFunc   ;==>_OL_ViewApply

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewCreate
; Description ...: Creates a new view for the specified folder.
; Syntax.........: _OL_ViewCreate($oOL, $sView, $vFolder[, $iType = Default[, $iSaveOption = Default]])
; Parameters ....: $oOL         - Outlook object returned by a preceding call to _OL_Open()
;                  $sView       - Name of the new view
;                  $vFolder     - Folder object as returned by _OL_FolderAccess or full name of folder
;                  $iType       - [optional] The type of the new view. Can be any of the OlViewType enumeration (default = $olTableView)
;                  $iSaveOption - [optional] Specifies the permissions of the new view. Can be any of the OlViewSaveOption enumeration (default = $olViewSaveOptionThisFolderOnlyMe)
; Return values .: Success - Object of the created view
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vFolder has not been specified
;                  |2 - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3 - Error accessing the folders views. @extended is set to the COM error code
;                  |4 - Error creating the new view. @extended is set to the COM error code
;                  |5 - View already exists or an error occurred. @extended is set to the COM error code (if any)
;                  |6 - Error saving the new view. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewCreate($oOL, $sView, $vFolder, $iType = Default, $iSaveOption = Default)
	Local $oView, $oViews
	If $iType = Default Then $iType = $olTableView
	If $iSaveOption = Default Then $iSaveOption = $olViewSaveOptionThisFolderOnlyMe
	If Not IsObj($vFolder) Then
		If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
		Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
		If @error Then Return SetError(2, @error, 0)
		$vFolder = $aFolder[1]
	EndIf
	$oViews = $vFolder.Views
	If @error Then Return SetError(3, @error, 0)
	; Check if the view already exist
	If IsObj($oViews.Item($sView)) Then Return SetError(5, @error, 0)
	; Create view
	$oView = $oViews.Add($sView, $iType, $iSaveOption)
	If @error Then Return SetError(4, @error, 0)
	; Save view
	$oView.Save()
	If @error Then Return SetError(6, @error, 0)
	Return $oView
EndFunc   ;==>_OL_ViewCreate

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewDelete
; Description ...: Deletes a view from the specified folder.
; Syntax.........: _OL_ViewDelete($oOL, $vView[, $vFolder = Default] )
; Parameters ....: $oOL     - Outlook object returned by a preceding call to _OL_Open()
;                  $vView   - Index, name or object of the view to be deleted
;                  $vFolder - [optional] Folder object as returned by _OL_FolderAccess or full name of folder. Has to be specified when $vView is not a view object
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vFolder has not been specified
;                  |2 - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3 - Error accessing the folders views. @extended is set to the COM error code
;                  |4 - Error deleting the view. @extended is set to the COM error code
;                  |5 - View does not exist or an error occurred. @extended is set to the COM error code (if any)
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewDelete($oOL, $vView, $vFolder = Default)
	Local $oView, $oViews
	If IsObj($vView) Then
		$oView = $vView
	Else
		If Not IsObj($vFolder) Then
			If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
			Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
			If @error Then Return SetError(2, @error, 0)
			$vFolder = $aFolder[1]
		EndIf
		$oViews = $vFolder.Views
		If @error Then Return SetError(3, @error, 0)
		; Check if the view exist
		$oView = $oViews.Item($vView)
		If @error Or (Not IsObj($oView)) Then Return SetError(5, @error, 0)
	EndIf
	; Delete view
	$oView.Delete()
	If @error Then Return SetError(4, @error, 0)
	Return 1
EndFunc   ;==>_OL_ViewDelete

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewExists
; Description ...: Returns True if the specified view exists.
; Syntax.........: _OL_ViewExists($oOL, $vView, $vFolder)
; Parameters ....: $oOL     - Outlook object returned by a preceding call to _OL_Open()
;                  $vView   - Index or name of the view
;                  $vFolder - Folder object as returned by _OL_FolderAccess or full name of folder
; Return values .: Success - True
;                  Failure - Returns False and sets @error:
;                  |1 - $vFolder has not been specified
;                  |2 - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3 - Error accessing the folders views. @extended is set to the COM error code
;                  |4 - View does not exist or an error occurred. @extended is set to the COM error code (if any)
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewExists($oOL, $vView, $vFolder)
	Local $oView, $oViews
	If Not IsObj($vFolder) Then
		If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, False)
		Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
		If @error Then Return SetError(2, @error, False)
		$vFolder = $aFolder[1]
	EndIf
	$oViews = $vFolder.Views
	If @error Then Return SetError(3, @error, False)
	; Check if the view exist
	$oView = $oViews.Item($vView)
	If @error Or (Not IsObj($oView)) Then Return SetError(4, @error, False)
	Return True
EndFunc   ;==>_OL_ViewExists

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewFieldAdd
; Description ...: Adds/inserts fields to/into a view and sets the formatting.
; Syntax.........: _OL_ViewFieldAdd($oOL, $vView, $vFolder = Default, $aFields)
; Parameters ....: $oOL     - Outlook object returned by a preceding call to _OL_Open()
;                  $vView   - Index, name or object of the view to be modified
;                  $vFolder - Folder object as returned by _OL_FolderAccess or full name of folder. Has to be specified when $vView is not a view object
;                  $aFields - two dimensional zero-based array describing the ViewFields to be added/inserted with the following columns:
;                  |0 - Name of the property (field name: displayed in the Field Chooser) to be added/inserted
;                  |1 - One-based index number where to insert the property. If empty or 0 the property will be added after the last property
;                  |2 - Align: Alignment of the ViewField column. Can be any of the OlAlign enumeration
;                  |3 - FieldFormat: Display format of the property depending on the FieldType. For details please see Remarks
;                  |4 - Label: Column label and tooltip displayed for the property
;                  |5 - Width: Approximate width (in characters) of the column
;                  $bReset  - [optional] Removes all existing ViewFields from the View before adding the new ViewFields (default = False)
; Return values .: Success - Object of the view
;                  Failure - Returns 0 and sets @error:
;                  |1  - $vFolder has not been specified
;                  |2  - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3  - Error accessing the folders views. @extended is set to the COM error code
;                  |4  - View does not exist or an error occurred. @extended is set to the COM error code (if any)
;                  |5  - Error adding/inserting a ViewField. @extended is set to the zero-based index of the ViewField in error
;                  |6  - Error setting the Align ColumnFormat parameter for the specified ViewField. @extended is set to the zero-based index of the ViewField in error
;                  |7  - Error setting the FieldFormat ColumnFormat parameter for the specified ViewField. @extended is set to the zero-based index of the ViewField in error
;                  |8  - Error setting the Label ColumnFormat parameter for the specified ViewField. @extended is set to the zero-based index of the ViewField in error
;                  |9  - Error setting the Width ColumnFormat parameter for the specified ViewField. @extended is set to the zero-based index of the ViewField in error
;                  |10 - Error saving the new view. @extended is set to the COM error code
;                  |11 - Invalid ViewType. Fields can only be added to CardViews and TableViews
; Author ........: water
; Modified ......:
; Remarks .......: FieldFormat: https://docs.microsoft.com/en-us/office/vba/api/outlook.columnformat.fieldformat
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewFieldAdd($oOL, $vView, $vFolder, $aFields, $bReset = False)
	Local $oView, $oViews, $oViewField
	If $bReset = Default Then $bReset = False
	If IsObj($vView) Then
		$oView = $vView
	Else
		If Not IsObj($vFolder) Then
			If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
			Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
			If @error Then Return SetError(2, @error, 0)
			$vFolder = $aFolder[1]
		EndIf
		$oViews = $vFolder.Views
		If @error Then Return SetError(3, @error, 0)
		; Check if the view exist
		$oView = $oViews.Item($vView)
		If @error Or (Not IsObj($oView)) Then Return SetError(4, @error, 0)
	EndIf
	If $oView.ViewType <> $olCardView And $oView.ViewType <> $olTableView Then Return SetError(11, 0, 0)
	If $bReset Then ; Remove the first VieField until all have been removed
		With $oView.ViewFields
			While 1
				.Remove(1)
				If @error Then ExitLoop
			WEnd
		EndWith
	EndIf
	; Add fields
	For $i = 0 To UBound($aFields) - 1
		If IsNumber($aFields[$i][1]) And Number($aFields[$i][1]) > 0 Then
			$oViewField = $oView.ViewFields.Insert($aFields[$i][0], Number($aFields[$i][1])) ; Fieldname
			If @error Or (Not IsObj($oViewField)) Then Return SetError(5, @error, 0)
		Else
			$oViewField = $oView.ViewFields.Add($aFields[$i][0]) ; Fieldname
			If @error Or (Not IsObj($oViewField)) Then Return SetError(5, @error, 0)
		EndIf
		$oViewField.ColumnFormat.Align = $aFields[$i][2]
		If @error Then Return SetError(6, @error, 0)
		$oViewField.ColumnFormat.FieldFormat = $aFields[$i][3]
		If @error Then Return SetError(7, @error, 0)
		$oViewField.ColumnFormat.Label = $aFields[$i][4]
		If @error Then Return SetError(8, @error, 0)
		$oViewField.ColumnFormat.Width = $aFields[$i][5]
		If @error Then Return SetError(9, @error, 0)
	Next
	; Save view
	$oView.Save
	If @error Then Return SetError(10, @error, 0)
	Return $oView
EndFunc   ;==>_OL_ViewFieldAdd

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewFontSet
; Description ...: Sets the properties of the ViewFont object for the specified View object.
; Syntax.........: _OL_ViewFontSet($oObject, $iSize = Default, $bBold = Default, $iColor = Default, $iExtendedColor = Default, $bItalic = Default, $bStrikethrough = Default, $bUnderline = Default)
; Parameters ....: $oObject        - Object to set the font properties for. Depending on the view type this can be one of the following:
;                  |BusinessCardView: The HeadingsFont property
;                  |CalendarView:     The DayWeekFont, DayWeekTimeFont or MonthFont property
;                  |CardView:         The BodyFont or HeadingsFont property
;                  |TableView:        The AutoPreviewFont, ColumnFont or RowFont property
;                  |TimelineView:     The ItemFont, LowerScaleFont or UpperScaleFont property
;                  $iSize          - [optional] Size (in points) of the font
;                  $bBold          - [optional] True if the font is bold
;                  $iColor         - [optional] Color of the font. Can be any of the OlColor enumeration
;                  $iExtendedColor - [optional] Extended color of the font. Can be any of the OlCategoryColor enumeration
;                  $bItalic        - [optional] If the font is italicized
;                  $bStrikethrough - [optional] If the font is struck through with a horizontal line
;                  $bUnderline     - [optional] If the font is underlined
; Return values .: Success - 1
;                  Failure - Returns 0 and sets @error:
;                  |1 - $oObject ist not an object type
;                  |2 - Error setting property Size of the object. @extended is set to the COM error code
;                  |3 - Error setting property Bold of the object. @extended is set to the COM error code
;                  |4 - Error setting property Color of the object. @extended is set to the COM error code
;                  |5 - Error setting property ExtendedColor of the object. @extended is set to the COM error code
;                  |6 - Error setting property Italic of the object. @extended is set to the COM error code
;                  |7 - Error setting property Strikethrough of the object. @extended is set to the COM error code
;                  |8 - Error setting property Underline of the object. @extended is set to the COM error code
;                  |9 - Error saving the new view. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......: The ViewFont object is used by the following objects to represent font formatting information applied to the text in various portions of a view:
;                  The HeadingsFont property of the BusinessCardView object.
;                  The DayWeekFont, DayWeekTimeFont, and MonthFont properties of the CalendarView object.
;                  The BodyFont and HeadingsFont properties of the CardView object.
;                  The AutoPreviewFont, ColumnFont, and RowFont properties of the TableView object.
;                  The ItemFont, LowerScaleFont, and UpperScaleFont properties of the TimelineView object.
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewFontSet($oObject, $iSize = Default, $bBold = Default, $iColor = Default, $iExtendedColor = Default, $bItalic = Default, $bStrikethrough = Default, $bUnderline = Default)
	If Not IsObj($oObject) Then Return SetError(1, 0, 0)
	If $iSize <> Default Then
		$oObject.Size = $iSize
		If @error Then Return SetError(2, @error, 0)
	EndIf
	If $bBold <> Default Then
		$oObject.Bold = $bBold
		If @error Then Return SetError(3, @error, 0)
	EndIf
	If $iColor <> Default Then
		$oObject.Color = $iColor
		If @error Then Return SetError(4, @error, 0)
	EndIf
	If $iExtendedColor <> Default Then
		$oObject.ExtendedColor = $iExtendedColor
		If @error Then Return SetError(5, @error, 0)
	EndIf
	If $bItalic <> Default Then
		$oObject.Italic = $bItalic
		If @error Then Return SetError(6, @error, 0)
	EndIf
	If $bStrikethrough <> Default Then
		$oObject.Strikethrough = $bStrikethrough
		If @error Then Return SetError(7, @error, 0)
	EndIf
	If $bUnderline <> Default Then
		$oObject.Underline = $bUnderline
		If @error Then Return SetError(8, @error, 0)
	EndIf
	; Save the view
	$oObject.Parent.Save()
	If @error Then Return SetError(9, @error, 0)
	Return 1
EndFunc   ;==>_OL_ViewFontSet

#Region NOCH OFFEN!!

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewFormatRuleAdd
; Description ...: Adds/inserts a new AutoFormat rule to a view.
; Syntax.........: _OL_ViewFormatRuleAdd($oOL, $vView, $vFolder, $sName, $sFilter, $iIndex = Default, $bEnabled = True)
; Parameters ....: $oOL      - Outlook object returned by a preceding call to _OL_Open()
;                  $vView    - Index, name or object of the view to be modified
;                  $vFolder  - Folder object as returned by _OL_FolderAccess or full name of folder. Has to be specified when $vView is not a view object, els the parameter is ignored
;                  $sName    - Display name for the view (default = do not modify)
;                  $sFilter  - [optional] String in DAV Searching and Locating (DASL) syntax that represents the current filter for the view (default = do not modify)
;                  $iIndex   - [optional] Either the index number at which to insert the rule, or the name of a rule in the collection where the new rule is to be inserted. See Remark
;                  $bEnabled - [optional] Indicates whether the formatting rule is enabled (default = True)
; Return values .: Success - Object of the Format Rule
;                  Failure - Returns 0 and sets @error:
;                  |1  - $vFolder has not been specified
; Author ........: water
; Modified ......:
; Remarks .......: $iIndex: When set to ...
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewFormatRuleAdd($oOL, $vView, $vFolder, $sName, $sFilter, $bEnabled = True)

EndFunc   ;==>_OL_ViewFormatRuleAdd

#EndRegion NOCH OFFEN!!

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewGet
; Description ...: Returns the object of the current view or a list of all views for the specified folder.
; Syntax.........: _OL_ViewGet($oOL, $vFolder[, $iFlag = 0])
; Parameters ....: $oOL     - Outlook object returned by a preceding call to _OL_Open()
;                  $vFolder - Folder object as returned by _OL_FolderAccess or full name of folder
;                  $iFlag   - [optional] One of the following flags:
;                  |0 - Return all views for the current folder in a 2D-array (default)
;                  |1 - Return the object of the current view for the specified folder
; Return values .: Success - When $iFlag = 0: One based two-dimensional array with the following properties:
;                  |0 - Name of the view
;                  |1 - Object of the view
;                  |2 - Type of the view object described by the OlViewType enumeration
;                  |3 - True if the view is a built-in Outlook view
;                  |4 - True if a user can modify the settings of the view
;                  |5 - Specifies the folders in which the specified view is available and the read permissions attached to the view. Described by the OlViewSaveOption enumeration
;                  |6 - Filter for the view in DAV Searching and Locating (DASL) syntax
;                  Success - When $iFlag = 1: Object of the current view for the folder
;                  Failure - Returns "" and sets @error:
;                  |1 - $vFolder has not been specified
;                  |2 - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3 - Error accessing the folders views. @extended is set to the COM error code
;                  |4 - Error accessing the current view of the specified folder. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewGet($oOL, $vFolder, $iFlag = 0)
	Local $oView, $oViews
	If $iFlag = Default Then $iFlag = 0
	If Not IsObj($vFolder) Then
		If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, "")
		Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
		If @error Then Return SetError(2, @error, "")
		$vFolder = $aFolder[1]
	EndIf
	If $iFlag = 1 Then
		$oView = $vFolder.CurrentView
		If @error Then Return SetError(4, @error, "")
		Return $oView
	EndIf
	$oViews = $vFolder.Views
	If @error Then Return SetError(3, @error, "")
	Local $aViews[$oViews.Count + 1][7] = [[$oViews.Count, 7]]
	For $i = 1 To $oViews.Count
		$aViews[$i][0] = $oViews.Item($i).Name
		$aViews[$i][1] = $oViews.Item($i)
		$aViews[$i][2] = $oViews.Item($i).ViewType
		$aViews[$i][3] = $oViews.Item($i).Standard
		$aViews[$i][4] = $oViews.Item($i).LockUserChanges
		$aViews[$i][5] = $oViews.Item($i).SaveOption
		$aViews[$i][6] = $oViews.Item($i).Filter
	Next
	Return $aViews
EndFunc   ;==>_OL_ViewGet

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewModify
; Description ...: Modifies properties of a view.
; Syntax.........: _OL_ViewModify($oOL, $vView, [$vFolder = Default[, $sName = Default[, $bLockUserChanges = Default[, $sFilter = Default[, $sXML = Default]]]]])
; Parameters ....: $oOL              - Outlook object returned by a preceding call to _OL_Open()
;                  $vView            - Index, name or object of the view to be modified
;                  $vFolder          - [optional] Folder object as returned by _OL_FolderAccess or full name of folder. Has to be specified when $vView is not a view object
;                  $sName            - [optional] Display name for the view (default = do not modify)
;                  $sLanguage        - [optional] String that represents the language setting for the object that defines the language used in the menu (default = do not modify)
;                  $bLockUserChanges - [optional] Indicates whether a user can modify the settings of the current view. Use True or False (default = do not modify)
;                  $sFilter          - [optional] String in DAV Searching and Locating (DASL) syntax that represents the current filter for the view (default = do not modify)
;                  $sXML             - [optional] String value that specifies the XML definition of the current view (default = do not modify)
; Return values .: Success - Object of the view
;                  Failure - Returns 0 and sets @error:
;                  |1  - $vFolder has not been specified
;                  |2  - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3  - Error accessing the folders views. @extended is set to the COM error code
;                  |4  - View does not exist or an error occurred. @extended is set to the COM error code (if any)
;                  |5  - Error modifying property "Name". @extended is set to the COM error code
;                  |6  - Error modifying property "Language". @extended is set to the COM error code
;                  |7  - Error modifying property "LockUserChanges". @extended is set to the COM error code
;                  |8  - Error modifying property "Filter". @extended is set to the COM error code
;                  |9  - Error modifying property "XML". @extended is set to the COM error code
;                  |10 - Error saving the new view. @extended is set to the COM error code
; Author ........: water
; Modified ......:
; Remarks .......: $sLanguage: Example: "EN-US" represents the ISO code for "United States - English"
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewModify($oOL, $vView, $vFolder = Default, $sName = Default, $sLanguage = Default, $bLockUserChanges = Default, $sFilter = Default, $sXML = Default)
	Local $oView, $oViews
	If IsObj($vView) Then
		$oView = $vView
	Else
		If Not IsObj($vFolder) Then
			If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
			Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
			If @error Then Return SetError(2, @error, 0)
			$vFolder = $aFolder[1]
		EndIf
		$oViews = $vFolder.Views
		If @error Then Return SetError(3, @error, 0)
		; Check if the view exist
		$oView = $oViews.Item($vView)
		If @error Or (Not IsObj($oView)) Then Return SetError(4, @error, 0)
	EndIf
	; Modify view
	If $sName <> Default And $sName <> "" Then
		$oView.Name = $sName
		If @error Then Return SetError(5, @error, 0)
	EndIf
	If $sLanguage <> Default And $sLanguage <> "" Then
		$oView.Language = $sLanguage
		If @error Then Return SetError(6, @error, 0)
	EndIf
	If $bLockUserChanges <> Default Then
		$oView.LockUserChanges = $bLockUserChanges
		If @error Then Return SetError(7, @error, 0)
	EndIf
	If $sFilter <> Default And $sFilter <> "" Then
		$oView.Filter = $sFilter
		If @error Then Return SetError(8, @error, 0)
	EndIf
	If $sXML <> Default And $sXML <> "" Then
		$oView.XML = $sXML
		If @error Then Return SetError(9, @error, 0)
	EndIf
	; Save view
	$oView.Save()
	If @error Then Return SetError(10, @error, 0)
	Return $oView
EndFunc   ;==>_OL_ViewModify

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewPropertyGet
; Description ...: Returns all properties of a view.
; Syntax.........: _OL_ViewPropertyGet($oOL, $vView[, $vFolder])
; Parameters ....: $oOL     - Outlook object returned by a preceding call to _OL_Open()
;                  $vView   - Index, name or object of the view to be queried
;                  $vFolder - [optional] Folder object as returned by _OL_FolderAccess or full name of folder. Has to be specified when $vView is not a view object, els the parameter is ignored
; Return values .: Success - two-dimensional one based array with the following information:
;                  |0 - Name of the property
;                  |1 - Value of the property
;                  |2 - Type of the property (1 if the property value is an object, else 0. So you can distinguish from an empty value)
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vFolder has not been specified
;                  |2 - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3 - Error accessing the folders views. @extended is set to the COM error code
;                  |4 - View does not exist or an error occurred. @extended is set to the COM error code (if any)
;                  |5 - ViewType of the specified view can not be found in table $aViewProperties
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewPropertyGet($oOL, $vView, $vFolder = Default)
	Local $oView, $oViews, $iViewType
	Local $bFound = False, $aProperties, $sCommand
	Local $aViewProperties[][] = [ _
			[$olBusinessCardView, "CardSize,CardSize,Class,Filter,HeadingsFont,Language,LockUserChanges,Name,Parent,SaveOption,SortFields,Standard,ViewType,XML"], _
			[$olCalendarView, "AutoFormatRules,BoldDatesWithItems,BoldSubjects,CalendarViewMode,Class,DaysInMultiDayMode,DayWeekTimeScale,DisplayedDates," & _
			"EndField,Filter,Language,LockUserChanges,MonthShowEndTime,Name,Parent,SaveOption,SelectedEndTime,SelectedStartTime,Standard,StartField,ViewType,XML"], _
			[$olCardView, "AllowInCellEditing,AutoFormatRules,BodyFont,Class,Filter,HeadingsFont,Language,LockUserChanges,MultiLineFieldHeight,Name,Parent,SaveOption," & _
			"ShowEmptyFields,SortFields,Standard,ViewFields,ViewType,Width,XML"], _
			[$olIconView, "Class,Filter,IconPlacement,IconViewType,Language,LockUserChanges,Name,Parent,SaveOption,SortFields,Standard,ViewType,XML"], _
			[$olPeopleView, "Class,Filter,Language,LockUserChanges,Name,Parent,SaveOption,SortFields,Standard,ViewType,XML"], _
			[$olTableView, "AllowInCellEditing,AlwaysExpandConversation,AutoFormatRules,AutomaticColumnSizing,AutomaticGrouping,AutoPreview,AutoPreviewFont,Class,ColumnFont," & _
			"DefaultExpandCollapseSetting,Filter,GridLineStyle,GroupByFields,HideReadingPaneHeaderInfo,Language,LockUserChanges,MaxLinesInMultiLineView,MultiLine," & _
			"MultiLineWidth,Name,Parent,RowFont,SaveOption,ShowConversationByDate,ShowConversationSendersAboveSubject,ShowFullConversations,ShowItemsInGroups,ShowNewItemRow," & _
			"ShowReadingPane,SortFields,Standard,ViewFields,ViewType,XML"], _
			[$olTimelineView, "Class,DefaultExpandCollapseSetting,EndField,Filter,GroupByFields,ItemFont,Language,LockUserChanges,LowerScaleFont,MaxLabelWidth,Parent,SaveOption," & _
			"ShowLabelWhenViewingByMonth,ShowWeekNumbers,Standard,StartField,TimelineViewMode,UpperScaleFont,ViewType,XML"] _
			]
	If IsObj($vView) Then
		$oView = $vView
	Else
		If Not IsObj($vFolder) Then
			If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
			Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
			If @error Then Return SetError(2, @error, 0)
			$vFolder = $aFolder[1]
		EndIf
		$oViews = $vFolder.Views
		If @error Then Return SetError(3, @error, 0)
		; Check if the view exist
		$oView = $oViews.Item($vView)
		If @error Or (Not IsObj($oView)) Then Return SetError(4, @error, 0)
	EndIf
	$iViewType = $oView.ViewType
	; Check if $aViewProperties holds valid properties for the view type
	For $i = 0 To UBound($aViewProperties) - 1
		If $aViewProperties[$i][0] = $iViewType Then
			$bFound = True
			ExitLoop
		EndIf
	Next
	If $bFound = False Then Return SetError(5, 0, 0)
	$aProperties = StringSplit($aViewProperties[$i][1], ",", $STR_NOCOUNT)
	Local $aResult[UBound($aProperties) + 1][3] = [[UBound($aProperties), 3]]
	For $j = 0 To UBound($aProperties) - 1
		$sCommand = '$oView.' & $aProperties[$j]
		$aResult[$j + 1][0] = $aProperties[$j]
		$aResult[$j + 1][1] = Execute($sCommand)
		$aResult[$j + 1][2] = IsObj($aResult[$j][1])
	Next
	Return $aResult
EndFunc   ;==>_OL_ViewPropertyGet

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewPropertySet
; Description ...: Modifies properties of a view.
; Syntax.........: _OL_ViewPropertySet($oOL, $vView, $vFolder, $aProperties)
; Parameters ....: $oOL         - Outlook object returned by a preceding call to _OL_Open()
;                  $vView       - Index, name or object of the view to be modified
;                  $vFolder     - Folder object as returned by _OL_FolderAccess or full name of folder. Has to be specified when $vView is not a view object, els the parameter is ignored
;                  $aProperties - One dimensional zero-based array holding the properties to set based on the view type. Format: propertyname=propertyvalue
; Return values .: Success - Object of the view
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vFolder has not been specified
;                  |2 - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3 - Error accessing the folders views. @extended is set to the COM error code
;                  |4 - View does not exist or an error occurred. @extended is set to the COM error code (if any)
;                  |5 - Invalid format of property in $aProperties. @extended is set to the zero-based index of the property in error
;                  |6 - Invalid property name for the view type. @extended is set to the zero-based index of the property in error
;                  |7 - Error saving the new view. @extended is set to the COM error code
;                  |8 - Error setting the property. @extended is set to the zero-based index of the property in error
; Author ........: water
; Modified ......:
; Remarks .......: Empty rows in $aProperties are ignored
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewPropertySet($oOL, $vView, $vFolder, $aProperties)
	Local $aViewProperties[][] = [ _
			[$olBusinessCardView, "CardSize"], _
			[$olBusinessCardView, "XML"], _
			[$olCalendarView, "BoldDatesWithItems"], _
			[$olCalendarView, "BoldSubjects"], _
			[$olCalendarView, "CalendarViewMode"], _
			[$olCalendarView, "DaysInMultiDayMode"], _
			[$olCalendarView, "DayWeekTimeScale"], _
			[$olCalendarView, "DisplayedDates"], _
			[$olCalendarView, "EndField"], _
			[$olCalendarView, "MonthShowEndTime"], _
			[$olCalendarView, "SelectedEndTime"], _
			[$olCalendarView, "SelectedStartTime"], _
			[$olCalendarView, "StartField"], _
			[$olCalendarView, "XML"], _
			[$olCardView, "AllowInCellEditing"], _
			[$olCardView, "MultiLineFieldHeight"], _
			[$olCardView, "ShowEmptyFields"], _
			[$olCardView, "Width"], _
			[$olCardView, "XML"], _
			[$olIconView, "IconPlacement"], _
			[$olIconView, "IconViewType"], _
			[$olIconView, "XML"], _
			[$olPeopleView, "XML"], _
			[$olTableView, "AllowInCellEditing"], _
			[$olTableView, "AlwaysExpandConversation"], _
			[$olTableView, "AutomaticColumnSizing"], _
			[$olTableView, "AutomaticGrouping"], _
			[$olTableView, "AutoPreview"], _
			[$olTableView, "DefaultExpandCollapseSetting"], _
			[$olTableView, "GridLineStyle"], _
			[$olTableView, "HideReadingPaneHeaderInfo"], _
			[$olTableView, "MaxLinesInMultiLineView"], _
			[$olTableView, "MultiLine"], _
			[$olTableView, "MultiLineWidth"], _
			[$olTableView, "ShowConversationByDate"], _
			[$olTableView, "ShowConversationSendersAboveSubject"], _
			[$olTableView, "ShowFullConversations"], _
			[$olTableView, "ShowItemsInGroups"], _
			[$olTableView, "ShowNewItemRow"], _
			[$olTableView, "ShowReadingPane"], _
			[$olTableView, "XML"], _
			[$olTimelineView, "DefaultExpandCollapseSetting"], _
			[$olTimelineView, "EndField"], _
			[$olTimelineView, "MaxLabelWidth"], _
			[$olTimelineView, "ShowLabelWhenViewingByMonth"], _
			[$olTimelineView, "ShowWeekNumbers"], _
			[$olTimelineView, "StartField"], _
			[$olTimelineView, "TimelineViewMode"], _
			[$olTimelineView, "UpperScaleFont"], _
			[$olTimelineView, "XML"] _
			]
	Local $bFound = False, $aProperty, $oView, $oViews, $iViewType
	If IsObj($vView) Then
		$oView = $vView
	Else
		If Not IsObj($vFolder) Then
			If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
			Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
			If @error Then Return SetError(2, @error, 0)
			$vFolder = $aFolder[1]
		EndIf
		$oViews = $vFolder.Views
		If @error Then Return SetError(3, @error, 0)
		; Check if the view exist
		$oView = $oViews.Item($vView)
		If @error Or (Not IsObj($oView)) Then Return SetError(4, @error, 0)
	EndIf
	$iViewType = $oView.ViewType
	; Check if $aProperties holds valid properties for the view type
	For $i = 0 To UBound($aProperties) - 1
		$aProperties[$i] = StringStripWS($aProperties[$i], BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING, $STR_STRIPSPACES))
		$aProperty = StringSplit($aProperties[$i], "=", $STR_NOCOUNT)
		If @error And $aProperties[$i] <> "" Then Return SetError(5, $i, 0)
		$bFound = False
		For $j = 0 To UBound($aViewProperties) - 1
			If $aViewProperties[$j][0] = $iViewType And $aViewProperties[$j][1] = $aProperty[0] Then
				$bFound = True
				ExitLoop
			EndIf
		Next
		If $bFound = False Then Return SetError(6, $i, 0)
		; Set the property
		Switch $aProperty[0]
			Case "AllowInCellEditing"
				$oView.AllowInCellEditing = $aProperty[1]
			Case "AlwaysExpandConversation"
				$oView.AlwaysExpandConversation = $aProperty[1]
			Case "AutomaticColumnSizing"
				$oView.AutomaticColumnSizing = $aProperty[1]
			Case "AutomaticGrouping"
				$oView.AutomaticGrouping = $aProperty[1]
			Case "AutoPreview"
				$oView.AutoPreview = $aProperty[1]
			Case "BoldDatesWithItems"
				$oView.BoldDatesWithItems = $aProperty[1]
			Case "BoldSubjects"
				$oView.BoldSubjects = $aProperty[1]
			Case "CalendarViewMode"
				$oView.CalendarViewMode = $aProperty[1]
			Case "CardSize"
				$oView.CardSize = $aProperty[1]
			Case "DaysInMultiDayMode"
				$oView.DaysInMultiDayMode = $aProperty[1]
			Case "DayWeekTimeScale"
				$oView.DayWeekTimeScale = $aProperty[1]
			Case "DefaultExpandCollapseSetting"
				$oView.DefaultExpandCollapseSetting = $aProperty[1]
			Case "DisplayedDates"
				$oView.DisplayedDates = $aProperty[1]
			Case "EndField"
				$oView.EndField = $aProperty[1]
			Case "GridLineStyle"
				$oView.GridLineStyle = $aProperty[1]
			Case "HideReadingPaneHeaderInfo"
				$oView.HideReadingPaneHeaderInfo = $aProperty[1]
			Case "IconPlacement"
				$oView.IconPlacement = $aProperty[1]
			Case "IconViewType"
				$oView.IconViewType = $aProperty[1]
			Case "MaxLinesInMultiLineView"
				$oView.MaxLinesInMultiLineView = $aProperty[1]
			Case "MonthShowEndTime"
				$oView.MonthShowEndTime = $aProperty[1]
			Case "MultiLine"
				$oView.MultiLine = $aProperty[1]
			Case "MultiLineFieldHeight"
				$oView.MultiLineFieldHeight = $aProperty[1]
			Case "MultiLineWidth"
				$oView.MultiLineWidth = $aProperty[1]
			Case "SelectedEndTime"
				$oView.SelectedEndTime = $aProperty[1]
			Case "SelectedStartTime"
				$oView.SelectedStartTime = $aProperty[1]
			Case "ShowConversationByDate"
				$oView.ShowConversationByDate = $aProperty[1]
			Case "ShowConversationSendersAboveSubject"
				$oView.ShowConversationSendersAboveSubject = $aProperty[1]
			Case "ShowEmptyFields"
				$oView.ShowEmptyFields = $aProperty[1]
			Case "ShowFullConversations"
				$oView.ShowFullConversations = $aProperty[1]
			Case "ShowItemsInGroups"
				$oView.ShowItemsInGroups = $aProperty[1]
			Case "ShowNewItemRow"
				$oView.ShowNewItemRow = $aProperty[1]
			Case "ShowReadingPane"
				$oView.ShowReadingPane = $aProperty[1]
			Case "StartField"
				$oView.StartField = $aProperty[1]
			Case "Width"
				$oView.Width = $aProperty[1]
			Case "XML"
				$oView.XML = $aProperty[1]
		EndSwitch
		If @error Then Return SetError(8, $i, 0)
	Next
	; Save view
	$oView.Save()
	If @error Then Return SetError(7, @error, 0)
	Return $oView
EndFunc   ;==>_OL_ViewPropertySet

; #FUNCTION# ====================================================================================================================
; Name ..........: _OL_ViewSortFieldAdd
; Description ...: Adds/inserts SortFields for a view.
; Syntax.........: _OL_ViewSortFieldAdd($oOL, $vView, $vFolder = Default, $aSortFields)
; Parameters ....: $oOL     - Outlook object returned by a preceding call to _OL_Open()
;                  $vView   - Index, name or object of the view to be modified
;                  $vFolder - Folder object as returned by _OL_FolderAccess or full name of folder. Has to be specified when $vView is not a view object
;                  $aSortFields - two dimensional zero-based array describing the SortFields to be added/inserted with the following columns:
;                  |0 - Name of the property (field name: displayed in the Field Chooser) to sort on
;                  |1 - One-based index number where to insert the property. If empty or 0 the property will be added after the last property
;                  |2 - IsDescending: If set to True, the property is sorted in descending order (default = False - the property is sorted in ascending order
;                  $bReset  - [optional] Removes all existing SortFields from the View before adding the new SortFields (default = False)
; Return values .: Success - Object of the view
;                  Failure - Returns 0 and sets @error:
;                  |1 - $vFolder has not been specified
;                  |2 - Error accessing the specified folder. See @extended for errorcode returned by GetFolderFromID
;                  |3 - Error accessing the folders views. @extended is set to the COM error code
;                  |4 - View does not exist or an error occurred. @extended is set to the COM error code (if any)
;                  |5 - Error adding/inserting a SortField. @extended is set to the zero-based index of the SortField in error
;                  |6 - Error saving the new view. @extended is set to the COM error code
;                  |7 - Invalid ViewType. Fields can only be added to the following ViewTypes: BusinessCardView, CalendarView, CardView, IconView, PeopleView, TableView
; Author ........: water
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: Yes
; ===============================================================================================================================
Func _OL_ViewSortFieldAdd($oOL, $vView, $vFolder, $aSortFields, $bReset = False)
	Local $oView, $oViews, $oSortField
	If $bReset = Default Then $bReset = False
	If IsObj($vView) Then
		$oView = $vView
	Else
		If Not IsObj($vFolder) Then
			If StringStripWS($vFolder, BitOR($STR_STRIPLEADING, $STR_STRIPTRAILING)) = "" Then Return SetError(1, 0, 0)
			Local $aFolder = _OL_FolderAccess($oOL, $vFolder)
			If @error Then Return SetError(2, @error, 0)
			$vFolder = $aFolder[1]
		EndIf
		$oViews = $vFolder.Views
		If @error Then Return SetError(3, @error, 0)
		; Check if the view exist
		$oView = $oViews.Item($vView)
		If @error Or (Not IsObj($oView)) Then Return SetError(4, @error, 0)
	EndIf
	Switch $oView.ViewType
		Case $olBusinessCardView, $olCalendarView, $olCardView, $olIconView, $olPeopleView, $olTableView
		Case Else
			Return SetError(7, 0, 0)
	EndSwitch
	If $bReset Then $oView.SortFields.RemoveAll ; Remove all SortFields
	; Add order fields
	For $i = 0 To UBound($aSortFields) - 1
		If IsNumber($aSortFields[$i][1]) And Number($aSortFields[$i][1]) > 0 Then
			$oSortField = $oView.SortFields.Insert($aSortFields[$i][0], Number($aSortFields[$i][1]), $aSortFields[$i][2]) ; Fieldname, Index, IsDescending
			If @error Or (Not IsObj($oSortField)) Then Return SetError(5, @error, 0)
		Else
			$oSortField = $oView.SortFields.Add($aSortFields[$i][0], $aSortFields[$i][2]) ; Fieldname, IsDescending
			If @error Or (Not IsObj($oSortField)) Then Return SetError(5, @error, 0)
		EndIf
	Next
	; Save view
	$oView.Save
	If @error Then Return SetError(6, @error, 0)
	Return $oView
EndFunc   ;==>_OL_ViewSortFieldAdd
